<?php

namespace App\Http\Controllers;

use App\Models\InterviewAttempt;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class InterviewAttemptController extends Controller
{
    public function index(): JsonResponse
    {
        $interviews = InterviewAttempt::with(['user', 'job', 'application', 'interviewer'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $interviews,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $interview = InterviewAttempt::with(['user', 'job', 'application', 'interviewer'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $interview,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|integer|exists:users,User_id',
            'Job_id' => 'required|integer|exists:jobs,Job_id',
            'Application_id' => 'nullable|integer|exists:job_applications,Application_id',
            'Scheduled_at' => 'required|date_format:Y-m-d H:i:s|after:now',
            'Status' => 'required|in:scheduled,completed,cancelled,mock',
            'Is_completed' => 'sometimes|boolean',
            'Video_link' => 'nullable|url|max:255',
            'Duration_minutes' => 'sometimes|integer|min:1|max:240',
            'Interview_type' => 'required|in:technical,hr,managerial,mock',
            'Interviewer_id' => 'nullable|integer|exists:users,User_id',
            'Feedback' => 'nullable|string|max:2000',
            'integration_id' => 'nullable|string|max:100|alpha_num',
        ], [
            'User_id.required' => 'The user ID is required.',
            'User_id.integer' => 'The user ID must be a valid integer.',
            'User_id.exists' => 'The specified user ID does not exist.',
            'Job_id.required' => 'The job ID is required.',
            'Job_id.integer' => 'The job ID must be a valid integer.',
            'Job_id.exists' => 'The specified job ID does not exist.',
            'Application_id.integer' => 'The application ID must be a valid integer.',
            'Application_id.exists' => 'The specified application ID does not exist.',
            'Scheduled_at.required' => 'The scheduled date and time is required.',
            'Scheduled_at.date_format' => 'The scheduled date and time must be in the format YYYY-MM-DD HH:MM:SS.',
            'Scheduled_at.after' => 'The scheduled date and time must be in the future.',
            'Status.required' => 'The status is required.',
            'Status.in' => 'The status must be one of: scheduled, completed, cancelled, mock.',
            'Video_link.url' => 'The video link must be a valid URL.',
            'Duration_minutes.integer' => 'The duration must be a valid integer.',
            'Duration_minutes.min' => 'The duration must be at least 1 minute.',
            'Duration_minutes.max' => 'The duration cannot exceed 240 minutes.',
            'Interview_type.required' => 'The interview type is required.',
            'Interview_type.in' => 'The interview type must be one of: technical, hr, managerial, mock.',
            'Interviewer_id.integer' => 'The interviewer ID must be a valid integer.',
            'Interviewer_id.exists' => 'The specified interviewer ID does not exist.',
            'Feedback.max' => 'The feedback cannot exceed 2000 characters.',
            'integration_id.alpha_num' => 'The integration ID must contain only letters and numbers.',
            'integration_id.max' => 'The integration ID cannot exceed 100 characters.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $interview = InterviewAttempt::create($request->only([
            'User_id',
            'Job_id',
            'Application_id',
            'Scheduled_at',
            'Status',
            'Is_completed',
            'Video_link',
            'Duration_minutes',
            'Interview_type',
            'Interviewer_id',
            'Feedback',
            'integration_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $interview->load(['user', 'job', 'application', 'interviewer']),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $interview = InterviewAttempt::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|integer|exists:users,User_id',
            'Job_id' => 'sometimes|integer|exists:jobs,Job_id',
            'Application_id' => 'nullable|integer|exists:job_applications,Application_id',
            'Scheduled_at' => 'sometimes|date_format:Y-m-d H:i:s|after:now',
            'Status' => 'sometimes|in:scheduled,completed,cancelled,mock',
            'Is_completed' => 'sometimes|boolean',
            'Video_link' => 'nullable|url|max:255',
            'Duration_minutes' => 'sometimes|integer|min:1|max:240',
            'Interview_type' => 'sometimes|in:technical,hr,managerial,mock',
            'Interviewer_id' => 'nullable|integer|exists:users,User_id',
            'Feedback' => 'nullable|string|max:2000',
            'integration_id' => 'sometimes|nullable|string|max:100|alpha_num',
        ], [
            'User_id.integer' => 'The user ID must be a valid integer.',
            'User_id.exists' => 'The specified user ID does not exist.',
            'Job_id.integer' => 'The job ID must be a valid integer.',
            'Job_id.exists' => 'The specified job ID does not exist.',
            'Application_id.integer' => 'The application ID must be a valid integer.',
            'Application_id.exists' => 'The specified application ID does not exist.',
            'Scheduled_at.date_format' => 'The scheduled date and time must be in the format YYYY-MM-DD HH:MM:SS.',
            'Scheduled_at.after' => 'The scheduled date and time must be in the future.',
            'Status.in' => 'The status must be one of: scheduled, completed, cancelled, mock.',
            'Video_link.url' => 'The video link must be a valid URL.',
            'Duration_minutes.integer' => 'The duration must be a valid integer.',
            'Duration_minutes.min' => 'The duration must be at least 1 minute.',
            'Duration_minutes.max' => 'The duration cannot exceed 240 minutes.',
            'Interview_type.in' => 'The interview type must be one of: technical, hr, managerial, mock.',
            'Interviewer_id.integer' => 'The interviewer ID must be a valid integer.',
            'Interviewer_id.exists' => 'The specified interviewer ID does not exist.',
            'Feedback.max' => 'The feedback cannot exceed 2000 characters.',
            'integration_id.alpha_num' => 'The integration ID must contain only letters and numbers.',
            'integration_id.max' => 'The integration ID cannot exceed 100 characters.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $interview->update($request->only([
            'User_id',
            'Job_id',
            'Application_id',
            'Scheduled_at',
            'Status',
            'Is_completed',
            'Video_link',
            'Duration_minutes',
            'Interview_type',
            'Interviewer_id',
            'Feedback',
            'integration_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $interview->load(['user', 'job', 'application', 'interviewer']),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $interview = InterviewAttempt::findOrFail($id);
        $interview->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Interview attempt deleted successfully',
        ], 204);
    }
}
